from typing import Optional

class BettingStrategy:
    def __init__(self, initial_bankroll: float = 1000.0):
        self.bankroll = initial_bankroll
        
    def calculate_kelly_stake(self, probability: float, odds: float, fractional_kelly: float = 0.25) -> float:
        """
        Calculates the optimal stake percentage using the Kelly Criterion.
        Formula: f* = (bp - q) / b
        Where:
            f* is the fraction of the current bankroll to wager
            b is the net odds received on the wager (odds - 1)
            p is the probability of winning
            q is the probability of losing (1 - p)
        
        Args:
            probability (float): Model's estimated probability of winning (0.0 to 1.0)
            odds (float): Decimal odds offered by the bookmaker (e.g., 2.50)
            fractional_kelly (float): Conservative multiplier (default 0.25 for safety)
            
        Returns:
            float: Recommended stake as a percentage of bankroll (e.g., 0.02 for 2%)
        """
        if probability <= 0 or odds <= 1:
            return 0.0
            
        b = odds - 1
        q = 1 - probability
        
        f_star = (b * probability - q) / b
        
        # Apply fractional kelly for safety (full Kelly is very volatile)
        stake_percentage = f_star * fractional_kelly
        
        # Ensure stake is non-negative and capped (e.g., max 5% of bankroll)
        return max(0.0, min(stake_percentage, 0.05))

    def calculate_ev(self, probability: float, odds: float) -> float:
        """
        Calculates Expected Value (EV).
        EV = (Probability * Profit) - (Loss Probability * Stake)
        """
        if odds <= 1: return -1.0
        
        # Profit per unit stake would be (odds - 1)
        # Loss per unit stake is 1
        return (probability * (odds - 1)) - (1 - probability)

    def get_recommendation(self, probability: float, current_odds: Optional[float] = None) -> dict:
        """
        Generates a betting recommendation based on model probability and market odds.
        If odds are not provided, we estimate 'fair odds' and assume a small market margin.
        """
        fair_odds = 1 / probability if probability > 0 else 0
        
        # If no odds provided, we can't calculate EV or Kelly accurately against the market.
        # We return the fair odds as a benchmark.
        if current_odds is None:
            return {
                "fair_odds": round(fair_odds, 2),
                "expected_value": None,
                "kelly_stake_pct": None,
                "recommendation": "Monitor Odds"
            }
            
        ev = self.calculate_ev(probability, current_odds)
        kelly_pct = self.calculate_kelly_stake(probability, current_odds)
        
        rec = "NO BET"
        if ev > 0.02: # 2% edge minimum
            rec = "VALUE BET"
        if ev > 0.10:
            rec = "STRONG BET"
            
        return {
            "fair_odds": round(fair_odds, 2),
            "market_odds": current_odds,
            "expected_value": round(ev * 100, 2), # In percentage
            "kelly_stake_pct": round(kelly_pct * 100, 2), # In percentage
            "recommendation": rec
        }
