-- Criação do banco de dados (se necessário)
-- CREATE DATABASE IF NOT EXISTS madrugao;
-- USE madrugao;

-- Tabela de Usuários
CREATE TABLE IF NOT EXISTS users (
    id INT AUTO_INCREMENT PRIMARY KEY,
    username VARCHAR(50) NOT NULL UNIQUE,
    email VARCHAR(100) NOT NULL UNIQUE,
    password VARCHAR(255) NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Tabela de Roles (Papéis/Funções)
CREATE TABLE IF NOT EXISTS roles (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(50) NOT NULL UNIQUE,
    description VARCHAR(255)
);

-- Tabela de Permissions (Permissões)
CREATE TABLE IF NOT EXISTS permissions (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(50) NOT NULL UNIQUE,
    description VARCHAR(255)
);

-- Tabela Pivot: Usuários <-> Roles
CREATE TABLE IF NOT EXISTS user_roles (
    user_id INT NOT NULL,
    role_id INT NOT NULL,
    PRIMARY KEY (user_id, role_id),
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (role_id) REFERENCES roles(id) ON DELETE CASCADE
);

-- Tabela Pivot: Roles <-> Permissions
CREATE TABLE IF NOT EXISTS role_permissions (
    role_id INT NOT NULL,
    permission_id INT NOT NULL,
    PRIMARY KEY (role_id, permission_id),
    FOREIGN KEY (role_id) REFERENCES roles(id) ON DELETE CASCADE,
    FOREIGN KEY (permission_id) REFERENCES permissions(id) ON DELETE CASCADE
);

-- Inserção de dados iniciais (Seed)

-- Roles
INSERT INTO roles (name, description) VALUES 
('admin', 'Administrador do Sistema'),
('editor', 'Editor de Conteúdo'),
('user', 'Usuário Padrão')
ON DUPLICATE KEY UPDATE name=name;

-- Permissions
INSERT INTO permissions (name, description) VALUES 
('view_dashboard', 'Visualizar Dashboard'),
('manage_users', 'Gerenciar Usuários'),
('edit_posts', 'Editar Postagens')
ON DUPLICATE KEY UPDATE name=name;

-- Atribuir permissões às roles
-- Admin tem todas as permissões
INSERT INTO role_permissions (role_id, permission_id) 
SELECT r.id, p.id FROM roles r, permissions p WHERE r.name = 'admin'
ON DUPLICATE KEY UPDATE role_id=role_id;

-- Editor pode ver dashboard e editar posts
INSERT INTO role_permissions (role_id, permission_id) 
SELECT r.id, p.id FROM roles r, permissions p WHERE r.name = 'editor' AND p.name IN ('view_dashboard', 'edit_posts')
ON DUPLICATE KEY UPDATE role_id=role_id;

-- User pode apenas ver dashboard
INSERT INTO role_permissions (role_id, permission_id) 
SELECT r.id, p.id FROM roles r, permissions p WHERE r.name = 'user' AND p.name = 'view_dashboard'
ON DUPLICATE KEY UPDATE role_id=role_id;

-- Criar um usuário admin padrão (senha: admin123)
-- Hash gerado com password_hash('admin123', PASSWORD_DEFAULT)
INSERT INTO users (username, email, password) VALUES 
('admin', 'admin@example.com', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi')
ON DUPLICATE KEY UPDATE username=username;

-- Atribuir role admin ao usuário admin
INSERT INTO user_roles (user_id, role_id)
SELECT u.id, r.id FROM users u, roles r WHERE u.username = 'admin' AND r.name = 'admin'
ON DUPLICATE KEY UPDATE user_id=user_id;
